<?php
// --- CONFIGURACIÓN ---
// Define las configuraciones en un solo lugar para fácil mantenimiento.
define('UPLOAD_DIR', __DIR__ . '/../images/photos/');
define('MAX_FILE_SIZE', 1 * 1024 * 1024); // 1MB
define('ALLOWED_MIMES', [
    'image/jpeg',
    'image/png',
    'image/webp'
]);

// --- RESPUESTA JSON ESTANDARIZADA ---
// Una función para enviar respuestas JSON limpiamente.
function send_json_response($statusCode, $data) {
    header('Content-Type: application/json');
    http_response_code($statusCode);
    echo json_encode($data);
    exit();
}

// --- VALIDACIONES INICIALES ---

// 1. Solo aceptar peticiones POST.
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    send_json_response(405, ['success' => false, 'message' => 'Método no permitido.']);
}

// 2. Verificar que el archivo se haya subido y no haya errores de transferencia.
if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    $errorMessage = 'Error: ';
    switch ($_FILES['file']['error']) {
        case UPLOAD_ERR_INI_SIZE:
        case UPLOAD_ERR_FORM_SIZE:
            $errorMessage .= 'El archivo excede el tamaño máximo permitido.';
            break;
        case UPLOAD_ERR_NO_FILE:
            $errorMessage .= 'No se recibió ningún archivo.';
            break;
        default:
            $errorMessage .= 'Ocurrió un error inesperado en la subida.';
            break;
    }
    send_json_response(400, ['success' => false, 'message' => $errorMessage]);
}

// --- PROCESAMIENTO DEL ARCHIVO ---

try {
    $file_tmp_path = $_FILES['file']['tmp_name'];
    $file_size = $_FILES['file']['size'];
    // Usamos el nombre que ya viene único desde Node.js.
    $file_name = basename($_FILES['file']['name']);
    
    // 3. Validación de Tamaño (Lado del Servidor).
    if ($file_size > MAX_FILE_SIZE) {
        throw new Exception('El archivo es demasiado grande. El máximo permitido es de ' . (MAX_FILE_SIZE / 1024 / 1024) . ' MB.');
    }

    // 4. Validación de Tipo MIME (La más importante para seguridad).
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $file_mime_type = finfo_file($finfo, $file_tmp_path);
    finfo_close($finfo);

    if (!in_array($file_mime_type, ALLOWED_MIMES)) {
        throw new Exception('Tipo de archivo no válido. Se detectó: ' . $file_mime_type);
    }
    
    // 5. Construcción de Ruta Robusta y Mover Archivo.
    $destination_path = UPLOAD_DIR . $file_name;
    
    // Asegurarse de que el directorio de destino exista.
    if (!is_dir(UPLOAD_DIR)) {
        mkdir(UPLOAD_DIR, 0755, true);
    }

    if (!move_uploaded_file($file_tmp_path, $destination_path)) {
        // 6. Manejo de Errores Específico.
        throw new Exception('El servidor no pudo mover el archivo subido. Verifique los permisos de la carpeta.');
    }

    // --- ÉXITO ---
    send_json_response(200, ['success' => true, 'filename' => $file_name, 'path' => $destination_path]);

} catch (Exception $e) {
    // Capturar cualquier excepción de nuestras validaciones y enviar un error 400.
    send_json_response(400, ['success' => false, 'message' => $e->getMessage()]);
}

?>